<?php
/**
 * ===================================================================================
 * Admin Inbox Page (Mobile Responsive)
 * ===================================================================================
 *
 * File: inbox.php
 * Location: /homework_portal/admin/inbox.php
 *
 * Purpose:
 * 1. Displays messages in conversation threads with a responsive layout.
 * 2. Includes a "Delete Thread" button for each conversation.
 * 3. Displays feedback messages after an action is performed.
 */

require_once '../config.php';
require_once 'admin_helpers.php'; // Correctly include the helper file

// --- Security Check ---
// A general check to ensure an admin is logged in.
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit();
}

$success_message = '';
$error_message = '';
if (isset($_SESSION['success_message'])) {
    $success_message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
}
if (isset($_SESSION['error_message'])) {
    $error_message = $_SESSION['error_message'];
    unset($_SESSION['error_message']);
}


try {
    // Fetch all messages and order them to group conversations together.
    $sql = "
        SELECT
            tm.id, tm.title, tm.message, tm.created_at, tm.sender_type, tm.is_read, tm.file_path,
            COALESCE(tm.parent_message_id, tm.id) as thread_id,
            t.full_name as teacher_name
        FROM teacher_messages tm
        JOIN teachers t ON tm.teacher_id = t.id
        ORDER BY thread_id DESC, tm.created_at ASC
    ";
    $all_messages = $pdo->query($sql)->fetchAll();

    // Group messages into threads using PHP
    $threads = [];
    foreach ($all_messages as $message) {
        $threads[$message['thread_id']][] = $message;
    }

    // Mark all unread replies from teachers as 'read' for the logged-in admin
    $update_sql = "UPDATE teacher_messages SET is_read = 1 WHERE sender_type = 'teacher' AND is_read = 0";
    $pdo->query($update_sql);
    
    // Recalculate unread count for the sidebar badge (it will now be 0)
    $unread_count = 0;

} catch (PDOException $e) {
    $threads = [];
    $unread_count = 0; // Set to 0 on error
    $page_error = "Could not load messages. " . $e->getMessage();
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Inbox - Admin Portal</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600;700&family=Baloo+2:wght@700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Poppins', sans-serif; background-color: #f0f9ff; }
        .font-brand { font-family: 'Baloo 2', cursive; }
        .sidebar-link { display: flex; align-items: center; padding: 0.75rem 1rem; border-radius: 0.5rem; transition: background-color 0.2s, color 0.2s; }
        .sidebar-link:hover, .sidebar-link.active { background-color: #e0f2fe; color: #0284c7; }
        .notification-badge { display: inline-flex; align-items: center; justify-content: center; width: 22px; height: 22px; border-radius: 9999px; background-color: #ef4444; color: white; font-size: 0.75rem; font-weight: bold; }
        .message.teacher { background-color: #f0fdfa; border-color: #14b8a6; } /* teal */
        .message.admin { background-color: #eff6ff; border-color: #3b82f6; } /* blue */
    </style>
</head>
<body class="flex flex-col min-h-screen">
    <div class="relative min-h-screen md:flex">
        <!-- Mobile menu button -->
        <div class="md:hidden flex justify-between items-center bg-white p-4 shadow-md">
             <a href="dashboard.php">
                <h1 class="text-3xl font-bold text-red-600 font-sans">TTTEDDDEL</h1>
            </a>
            <button id="mobile-menu-button" class="text-gray-500 hover:text-blue-600 focus:outline-none">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16m-7 6h7"></path></svg>
            </button>
        </div>

        <!-- Sidebar Navigation -->
        <aside id="sidebar" class="w-64 bg-white shadow-md absolute inset-y-0 left-0 transform -translate-x-full md:relative md:translate-x-0 transition-transform duration-300 ease-in-out z-30">
            <div class="p-4 border-b text-center">
                <a href="dashboard.php">
                    <h1 class="text-4xl font-bold text-red-600 tracking-widest font-sans">TTTEDDDEL</h1>
                    <p class="text-xs font-semibold text-blue-800 mt-1">Teacher-Student Interactive System</p>
                    <p class="text-xs text-blue-800">by Ele Infocomm Technologies</p>
                </a>
            </div>
            <nav class="p-4 space-y-2">
                <a href="dashboard.php" class="sidebar-link"><span>Dashboard</span></a>
                <a href="inbox.php" class="sidebar-link active flex justify-between items-center">
                    <span>Inbox</span>
                    <?php if ($unread_count > 0): ?><span class="notification-badge"><?php echo $unread_count; ?></span><?php endif; ?>
                </a>
                <?php if (has_permission('manage_admins')): ?><a href="manage_admins.php" class="sidebar-link"><span>Manage Admins</span></a><?php endif; ?>
                <?php if (has_permission('manage_teachers')): ?><a href="manage_teachers.php" class="sidebar-link"><span>Manage Teachers</span></a><?php endif; ?>
                <?php if (has_permission('manage_classes')): ?><a href="manage_classes.php" class="sidebar-link"><span>Manage Classes</span></a><?php endif; ?>
                <?php if (has_permission('manage_subjects')): ?><a href="manage_subjects.php" class="sidebar-link"><span>Manage Subjects</span></a><?php endif; ?>
                <?php if (has_permission('assign_subjects')): ?><a href="assign_subjects.php" class="sidebar-link"><span>Assign Subjects</span></a><?php endif; ?>
                <?php if (has_permission('send_message')): ?><a href="send_message.php" class="sidebar-link"><span>Send Message</span></a><?php endif; ?>
            </nav>
        </aside>

        <!-- Main Content Area -->
        <div class="flex-1 flex flex-col overflow-hidden">
            <header class="bg-white shadow-sm border-b hidden md:flex"><div class="flex items-center justify-end p-4 w-full"><span class="text-gray-600 mr-4">Welcome, <?php echo e($_SESSION['admin_username'] ?? 'Admin'); ?>!</span><a href="account_settings.php" class="p-2 text-gray-500 hover:text-blue-600" title="Account Settings"><svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" /><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" /></svg></a><a href="logout.php" class="ml-2 py-2 px-4 bg-blue-600 text-white font-semibold rounded-lg shadow-md hover:bg-blue-700">Logout</a></div></header>

            <main class="flex-1 overflow-x-hidden overflow-y-auto bg-gray-100 p-4 md:p-6">
                <div class="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-6">
                    <h2 class="text-3xl font-semibold text-gray-800">Inbox</h2>
                    <a href="send_message.php" class="mt-4 sm:mt-0 w-full sm:w-auto px-6 py-2 bg-blue-600 text-white font-semibold rounded-lg shadow hover:bg-blue-700 text-center">Send New Message</a>
                </div>

                <?php if($success_message): ?><div class="p-4 my-4 text-sm text-green-800 bg-green-100 rounded-lg"><?php echo $success_message; ?></div><?php endif; ?>
                <?php if($error_message): ?><div class="p-4 my-4 text-sm text-red-800 bg-red-100 rounded-lg"><?php echo $error_message; ?></div><?php endif; ?>
                <?php if(isset($page_error)): ?><div class="p-4 my-4 text-sm text-red-800 bg-red-100 rounded-lg"><?php echo e($page_error); ?></div><?php endif; ?>

                <div class="mt-6 space-y-6">
                    <?php if(empty($threads)): ?>
                        <div class="bg-white p-6 rounded-lg shadow text-center text-gray-500">The inbox is empty.</div>
                    <?php else: ?>
                        <?php foreach($threads as $thread_id => $thread_messages): ?>
                            <div class="bg-white rounded-lg shadow-md overflow-hidden">
                                <div class="p-4 bg-gray-50 border-b flex flex-col sm:flex-row justify-between items-start sm:items-center">
                                    <div>
                                        <h3 class="font-bold text-lg text-gray-800"><?php echo e($thread_messages[0]['title']); ?></h3>
                                        <p class="text-sm text-gray-600">Conversation with: <?php echo e($thread_messages[0]['teacher_name']); ?></p>
                                    </div>
                                    <a href="message_actions.php?action=delete&id=<?php echo $thread_id; ?>" onclick="return confirm('Are you sure you want to delete this entire conversation? This action cannot be undone.');" class="mt-2 sm:mt-0 text-red-500 hover:text-red-700 text-sm font-medium">Delete Thread</a>
                                </div>
                                <div class="p-4 space-y-4">
                                    <?php foreach($thread_messages as $message): ?>
                                        <div class="message p-4 rounded-lg border-l-4 <?php echo $message['sender_type']; ?>">
                                            <div class="flex justify-between items-center text-xs text-gray-500 mb-2">
                                                <span class="font-semibold"><?php echo $message['sender_type'] === 'admin' ? 'You' : e($message['teacher_name']); ?></span>
                                                <span><?php echo date('M d, Y, g:i a', strtotime($message['created_at'])); ?></span>
                                            </div>
                                            <p class="text-gray-700 break-words"><?php echo nl2br(e($message['message'])); ?></p>
                                            <?php if (!empty($message['file_path'])): ?>
                                                <div class="mt-3">
                                                    <a href="../<?php echo e($message['file_path']); ?>" target="_blank" class="text-sm text-blue-600 hover:underline">
                                                        View Attachment
                                                    </a>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>
    <footer class="text-center py-6 bg-gray-100 border-t mt-auto">
        <p class="text-sm text-gray-500">Built By Ele InfoComm Technologies- Kenya | <a href="http://web.e-ict.co.ke" target="_blank" class="hover:underline">web.e-ict.co.ke</a> | <a href="mailto:info@e-ict.co.ke" class="hover:underline">info@e-ict.co.ke</a> | +254202000121 | +254 722 215810 | 2025 All right reserved.</p>
    </footer>

    <script>
        const btn = document.getElementById('mobile-menu-button');
        const sidebar = document.getElementById('sidebar');

        btn.addEventListener('click', () => {
            sidebar.classList.toggle('-translate-x-full');
        });
    </script>
</body>
</html>
